/* eslint global-require: off */

/**
 * This module executes inside of electron's main process. You can start
 * electron renderer process from here and communicate with the other processes
 * through IPC.
 *
 * When running `yarn build` or `yarn build-main`, this file is compiled to
 * `./app/main.prod.js` using webpack. This gives us some performance wins.
 *
 * @flow
 */
import { app, BrowserWindow, ipcMain, shell } from 'electron';
import path from 'path';
import MenuBuilder from './menu';
import { version, productName } from '../package.json';
import { setupLogger } from './library/Logger';
import { setupAutoUpdater } from './library/AutoUpdater';
import HeadsetWrapper from './library/headset/HeadsetWrapper';
import { IS_DEBUG, OS_IS_MAC } from './constants/versions';

if (process.env.NODE_ENV === 'development' || !process.env.NODE_ENV) {
    app.setPath(
        'userData',
        app.getPath('userData').replace(/Electron/i, productName)
    );
}

let mainWindow = null;
let headset = null; // eslint-disable-line
let wasAskedToClose = false;

const gotTheLock = app.requestSingleInstanceLock();

if (!gotTheLock) {
    app.quit();
}

if (process.env.NODE_ENV === 'production') {
    const sourceMapSupport = require('source-map-support');
    sourceMapSupport.install();
}

if (
    process.env.NODE_ENV === 'development' ||
    process.env.DEBUG_PROD === 'true'
) {
    require('electron-debug')();

    // When in debug mode, return correct version from package.json
    app.getVersion = () => version;
}

const installExtensions = async () => {
    const installer = require('electron-devtools-installer');
    const forceDownload = !!process.env.UPGRADE_EXTENSIONS;
    const extensions = ['REACT_DEVELOPER_TOOLS', 'REDUX_DEVTOOLS'];

    return Promise.all(
        extensions.map(name =>
            installer.default(installer[name], forceDownload)
        )
    ).catch(console.log);
};

/**
 * Add event listeners...
 */

ipcMain.on('minimize-window', () => {
    mainWindow.minimize();
});

ipcMain.on('maximize-window', () => {
    mainWindow.setFullScreen(!mainWindow.isFullScreen());
});

ipcMain.on('close-window', () => {
    app.quit();
});

ipcMain.on('QUIT_CONFIRM', () => {
    app.quit();
});

ipcMain.on('CANCEL_CONFIRM', () => {
    wasAskedToClose = false;
});

app.on('window-all-closed', () => {
    // Respect the OSX convention of having the application in memory even
    // after all windows have been closed

    if (process.platform !== 'darwin') {
        app.quit();
    }
    // Note: at this moment not closing directly is causing that application on OSX can't be closed on firmware update
    app.exit();
});

app.on('second-instance', (_event, _commandLine, _workingDirectory) => {
    // Someone tried to run a second instance, we should focus our window.
    if (mainWindow) {
        if (mainWindow.isMinimized()) mainWindow.restore();
        mainWindow.focus();
    }
});

app.on('ready', async () => {
    if (
        process.env.NODE_ENV === 'development' ||
        process.env.DEBUG_PROD === 'true'
    ) {
        await installExtensions();
    }

    setupLogger();
    mainWindow = new BrowserWindow({
        show: false,
        width: 960,
        height: OS_IS_MAC ? 621 : 681,
        titleBarStyle: OS_IS_MAC ? null : 'hidden',
        frame: !OS_IS_MAC,
        resizable: false,
        icon: path.join(__dirname, '../build/icon.ico')
    });

    mainWindow.loadURL(`file://${__dirname}/app.html`);

    // @TODO: Use 'ready-to-show' event
    //        https://github.com/electron/electron/blob/master/docs/api/browser-window.md#using-ready-to-show-event
    mainWindow.webContents.on('did-finish-load', () => {
        if (!mainWindow) {
            throw new Error('"mainWindow" is not defined');
        }
        if (process.env.START_MINIMIZED) {
            mainWindow.minimize();
        } else {
            mainWindow.show();
            mainWindow.focus();
        }

        headset = new HeadsetWrapper(ipcMain);
    });

    mainWindow.on('close', e => {
        if (wasAskedToClose) {
            app.quit();
        } else {
            e.preventDefault();
            mainWindow.webContents.send('windowApplicationClose');
            wasAskedToClose = true;
        }
    });

    mainWindow.on('closed', () => {
        mainWindow = null;
    });

    // Make sure that any links (inside the app) being clicked on, if they're pointing
    // to a new window - open them in the user's browser (not in the Electron window)
    mainWindow.webContents.on('new-window', (e, url) => {
        e.preventDefault();
        shell.openExternal(url);
    });

    if (IS_DEBUG) {
        const menuBuilder = new MenuBuilder(mainWindow);
        menuBuilder.buildMenu();
    }

    setupAutoUpdater(mainWindow);
});
